<?php

/*
 * TODO:
 * --------
 *  - (Possibly) assess memory optimizations
 */

class TestOfMessage extends UnitTestCase
{
	public function testMessageContainsNeededHeaders()
	{
		$msg = new Swift_Message("Test Subject", "my body");
		$structure = $msg->build()->readFull();
		$this->assertPattern(
			"~To: .*?\r\n".
			"From: .*?\r\n" .
			"Subject: .*?\r\n" .
			"Date: .*?\r\n" .
			"X-Mailer: .*?\r\n" .
			"MIME-Version: 1\\.0\r\n" .
			"Content-Type: .*?\r\n" .
			"Content-Transfer-Encoding: .*?\r\n\r\n~s", $structure);
		
		$msg = new Swift_Message();
		$structure = $msg->build()->readFull();
		$this->assertPattern(
			"~To: .*?\r\n".
			"From: .*?\r\n" .
			"Subject: .*?\r\n" .
			"Date: .*?\r\n" .
			"X-Mailer: .*?\r\n" .
			"MIME-Version: 1\\.0\r\n" .
			"Content-Type: .*?\r\n" .
			"Content-Transfer-Encoding: .*?\r\n\r\n~s", $structure);
	}
	
	public function testRecipientHeadersCanBeAListOfAddresses()
	{
		$headers = array("To", "Reply-To", "Cc");
		
		foreach ($headers as $field)
		{
			$method = "set" . str_replace("-", "", $field);
			$msg = new Swift_Message();
			$msg->$method("foo@bar.com");
			$structure = $msg->build()->readFull();
			$this->assertPattern(
				"~" . $field . ": foo@bar\\.com\r\n".
				".*?" .
				"Content-Transfer-Encoding: .*?\r\n\r\n~s", $structure);
			
			$msg->$method(array("joe@bloggs.tld", "Mr Grumpy <mr@grumpy.org>"));
			$structure = $msg->build()->readFull();
			$this->assertPattern(
				"~" . $field . ": joe@bloggs\\.tld,\\s*[\t ]Mr Grumpy <mr@grumpy\\.org>\r\n".
				".*?" .
				"Content-Transfer-Encoding: .*?\r\n\r\n~s", $structure);
				
			$msg = new Swift_Message("Some Subject");
			$msg->$method(array("joe@bloggs.tld", "Mr Grumpy <mr@grumpy.org>", "test"));
			$structure = $msg->build()->readFull();
			$this->assertPattern(
				"~" . $field . ": joe@bloggs\\.tld,\\s*[\t ]Mr Grumpy <mr@grumpy\\.org>,\\s*[\t ]test\r\n".
				".*?" .
				"Content-Transfer-Encoding: .*?\r\n\r\n~s", $structure);
		}
	}
	
	public function testUTF8CharsetIsUsedIfDetectedUnlessOverridden()
	{
		$msg = new Swift_Message();
		$msg->setBody("cenvéla");
		$structure = $msg->build()->readFull();
		$this->assertPattern("~Content-Type: text/plain;\\s* charset=utf-8~s", $structure);
		
		$msg = new Swift_Message();
		$msg->setBody("cenvela");
		$structure = $msg->build()->readFull();
		$this->assertPattern("~Content-Type: text/plain;\\s* charset=iso-8859-1~s", $structure);
	}
	
	public function testFlowedFormattingCanBeTurnedOnOrOff()
	{
		$msg = new Swift_Message();
		$msg->setBody("some body");
		$msg->setFlowed(true);
		$structure = $msg->build()->readFull();
		$this->assertPattern("~Content-Type: text/plain;\\s* charset=iso-8859-1;\\s* format=flowed~s", $structure);
		
		$msg = new Swift_Message();
		$msg->setBody("some body");
		$msg->setFlowed(false);
		$structure = $msg->build()->readFull();
		$this->assertNoPattern("~Content-Type: text/plain;\\s* charset=iso-8859-1;\\s* format=flowed~s", $structure);
	}
	
	public function testPriorityCanBeSet()
	{
		$msg = new Swift_Message();
		$msg->setPriority(1);
		$structure = $msg->build()->readFull();
		$this->assertPattern("~X-Priority: 1\r\nX-MSMail-Priority: High~s", $structure);
		
		$msg->setPriority(Swift_Message::PRIORITY_HIGH);
		$structure = $msg->build()->readFull();
		$this->assertPattern("~X-Priority: 1\r\nX-MSMail-Priority: High~s", $structure);
		
		$msg->setPriority(2);
		$structure = $msg->build()->readFull();
		$this->assertPattern("~X-Priority: 2\r\nX-MSMail-Priority: High~s", $structure);
		
		$msg->setPriority(3);
		$structure = $msg->build()->readFull();
		$this->assertPattern("~X-Priority: 3\r\nX-MSMail-Priority: Normal~s", $structure);
		
		$msg->setPriority(Swift_Message::PRIORITY_NORMAL);
		$structure = $msg->build()->readFull();
		$this->assertPattern("~X-Priority: 3\r\nX-MSMail-Priority: Normal~s", $structure);
		
		$msg->setPriority(4);
		$structure = $msg->build()->readFull();
		$this->assertPattern("~X-Priority: 4\r\nX-MSMail-Priority: Low~s", $structure);
		
		$msg->setPriority(Swift_Message::PRIORITY_LOW);
		$structure = $msg->build()->readFull();
		$this->assertPattern("~X-Priority: 5\r\nX-MSMail-Priority: Low~s", $structure);
		
		$msg->setPriority(5);
		$structure = $msg->build()->readFull();
		$this->assertPattern("~X-Priority: 5\r\nX-MSMail-Priority: Low~s", $structure);
	}
	
	public function testPriorityIsAdjustedIfSetTooHighOrLow()
	{
		for ($i = -10; $i < 1; $i++)
		{
			$msg = new Swift_Message();
			$msg->setPriority($i);
			$this->assertEqual(Swift_Message::PRIORITY_HIGH, $msg->getPriority());
			$structure = $msg->build()->readFull();
			$this->assertPattern("~X-Priority: 1\r\nX-MSMail-Priority: High~s", $structure);
		}
		
		for ($i = 15; $i > 5; $i--)
		{
			$msg = new Swift_Message();
			$msg->setPriority($i);
			$this->assertEqual(Swift_Message::PRIORITY_LOW, $msg->getPriority());
			$structure = $msg->build()->readFull();
			$this->assertPattern("~X-Priority: 5\r\nX-MSMail-Priority: Low~s", $structure);
		}
	}
	
	public function testReadReceiptHeadersAreSetIfRequested()
	{
		$msg = new Swift_Message();
		$msg->requestReadReceipt("foo@bar.tld");
		$structure = $msg->build()->readFull();
		$this->assertPattern("~Disposition-Notification-To: foo@bar\\.tld~s", $structure);
		
		$msg->requestReadReceipt("zip@button.tld");
		$structure = $msg->build()->readFull();
		$this->assertPattern("~Disposition-Notification-To: zip@button\\.tld~s", $structure);
		
		$msg->requestReadReceipt(false);
		$structure = $msg->build()->readFull();
		$this->assertNoPattern("~Disposition-Notification-To: .*~s", $structure);
	}
	
	public function testContentTypeIsSetToMultipartAlternativeIfpartsAreAdded()
	{
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_Part("foobar"));
		$structure = $msg->build()->readFull();
		$this->assertPattern("~(.(?!\r\n\r\n))*?\r\nContent-Type: multipart/alternative;\\s* boundary=.*~", $structure);
		
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_Part("foobar"));
		$msg->attach(new Swift_Message_Part("zipbutton", "text/html"));
		$structure = $msg->build()->readFull();
		$this->assertPattern("~(.(?!\r\n\r\n))*?\r\nContent-Type: multipart/alternative;\\s* boundary=.*~", $structure);
	}
	
	public function testBodyBecomesMimeWarningInMultipartMessages()
	{
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_Part("foobar"));
		$msg->attach(new Swift_Message_Part("zipbutton", "text/html"));
		$msg->setBody("This is a message in MIME format");
		$structure = $msg->build()->readFull();
		$this->assertPattern("~(.(?!\r\n\r\n))*?\r\nContent-Type: multipart/alternative;\\s* boundary=.*?\r\n\r\nThis is a message in MIME format\r\n--.*~s", $structure);
	}
	
	public function testMultipartAlternativeDocumentIsNestedCorrectly()
	{
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_Part("foobar"));
		$msg->attach(new Swift_Message_Part("zipbutton", "text/html"));
		$structure = $msg->build()->readFull();
		$this->assertPattern("~^(?:.(?!\r\n\r\n))*?\r\nContent-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\1\r\n" .
			".*?\r\n\r\n.*?\r\n--\\2\r\n" .
			".*?\r\n\r\nfoobar\r\n--\\2\r\n" .
			".*?\r\n\r\nzipbutton\r\n" .
			"--\\2--~s", $structure);
	}
	
	public function testHtmlPartsAlwaysAppearAfterPlainParts()
	{
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_Part("foobar"));
		$msg->attach(new Swift_Message_Part("zipbutton", "text/html"));
		$structure = $msg->build()->readFull();
		$this->assertPattern("~^(?:.(?!\r\n\r\n))*?\r\nContent-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\1\r\n" .
			".*?\r\n\r\n.*?\r\n--\\2\r\n" .
			".*?\r\n\r\nfoobar\r\n--\\2\r\n" .
			".*?\r\n\r\nzipbutton\r\n" .
			"--\\2--~s", $structure);
			
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_Part("zipbutton", "text/html"));
		$msg->attach(new Swift_Message_Part("foobar"));
		$structure = $msg->build()->readFull();
		$this->assertPattern("~^(?:.(?!\r\n\r\n))*?\r\nContent-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\1\r\n" .
			".*?\r\n\r\n.*?\r\n--\\2\r\n" .
			".*?\r\n\r\nfoobar\r\n--\\2\r\n" .
			".*?\r\n\r\nzipbutton\r\n" .
			"--\\2--~s", $structure);
	}
	
	public function testContentTypeIsSetToMultipartMixedIfAttachmentsAreAdded()
	{
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_Attachment("foobar"));
		$structure = $msg->build()->readFull();
		$this->assertPattern("~(.(?!\r\n\r\n))*?\r\nContent-Type: multipart/mixed;\\s* boundary=.*~", $structure);
		
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_Attachment("foobar"));
		$msg->attach(new Swift_Message_Attachment("zipbutton"));
		$structure = $msg->build()->readFull();
		$this->assertPattern("~(.(?!\r\n\r\n))*?\r\nContent-Type: multipart/mixed;\\s* boundary=.*~", $structure);
	}
	
	public function testContentTypeIsSetToMixedEvenIfPartsAreAddedWithAttachments()
	{
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_Attachment("foobar"));
		$msg->attach(new Swift_Message_Part("foo"));
		$structure = $msg->build()->readFull();
		$this->assertPattern("~(.(?!\r\n\r\n))*?\r\nContent-Type: multipart/mixed;\\s* boundary=.*~", $structure);
		
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_Part("some part"));
		$msg->attach(new Swift_Message_Part("another part"));
		$msg->attach(new Swift_Message_Attachment("foobar"));
		$msg->attach(new Swift_Message_Attachment("zipbutton"));
		$structure = $msg->build()->readFull();
		$this->assertPattern("~(.(?!\r\n\r\n))*?\r\nContent-Type: multipart/mixed;\\s* boundary=.*~", $structure);
	}
	
	//If regex is something which scares you, look away right now!
	public function testMessageIsCorrectlyStructuredForAttachmentsWithParts()
	{
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_Attachment("foobar"));
		$msg->attach(new Swift_Message_Part("foo"));
		$structure = $msg->build()->readFull();
		$this->assertPattern(
			"~.*?Content-Type: multipart/mixed;\\s* boundary=(\"?)(.*?)\\1\r\n" . //First headers
			".*?\r\n\r\n" . //End of first headers
			".*?\r\n--\\2\r\n" . //First mixed boundary
			"Content-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\3\r\n" . //Second headers
			".*?\r\n\r\n" . //End of second headers
			".*?\r\n--\\4\r\n" . //First alternative boundary
			".*?\r\n--\\4--" . //End of alternative parts
			"\\s*\r\n--\\2\r\n" . //Second mixed part
			".*?\r\n\r\n" .
			".*?\r\n--\\2--~s", //End of mixed parts
			$structure);
		
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_Part("some part"));
		$msg->attach(new Swift_Message_Part("another part"));
		$msg->attach(new Swift_Message_Attachment("foobar"));
		$msg->attach(new Swift_Message_Attachment("zipbutton"));
		$structure = $msg->build()->readFull();
		$this->assertPattern(
			"~.*?Content-Type: multipart/mixed;\\s* boundary=(\"?)(.*?)\\1\r\n" . //First headers
			".*?\r\n\r\n" . //End of first headers
			".*?\r\n--\\2\r\n" . //First mixed boundary
			"Content-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\3\r\n" . //Second headers
			".*?\r\n\r\n" . //End of second headers
			".*?\r\n--\\4\r\n" . //First alternative boundary
			".*?\r\n--\\4\r\n" . //Next alternative boundary
			".*?\r\n\r\n" .
			".*?\r\n--\\4--" . //End of alternative parts
			"\\s*\r\n--\\2\r\n" . //Second mixed part
			".*?\r\n\r\n" .
			".*?\r\n--\\2\r\n" . //Third mixed part
			".*?\r\n\r\n" .
			".*?\r\n--\\2--~s", //End of mixed parts
			$structure);
	}
	
	public function testMessageRemainsCorrectlyStructuredWhenPartsAreAddedAfterCompilation()
	{
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_Attachment("foobar"));
		$msg->attach(new Swift_Message_Part("foo"));
		$structure = $msg->build()->readFull();
		$this->assertPattern(
			"~.*?Content-Type: multipart/mixed;\\s* boundary=(\"?)(.*?)\\1\r\n" . //First headers
			".*?\r\n\r\n" . //End of first headers
			".*?\r\n--\\2\r\n" . //First mixed boundary
			"Content-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\3\r\n" . //Second headers
			".*?\r\n\r\n" . //End of second headers
			".*?\r\n--\\4\r\n" . //First alternative boundary
			".*?\r\n--\\4--" . //End of alternative parts
			"\\s*\r\n--\\2\r\n" . //Second mixed part
			".*?\r\n\r\n" .
			".*?\r\n--\\2--~s", //End of mixed parts
			$structure);
		$msg->attach(new Swift_Message_Part("bar"));
		$structure = $msg->build()->readFull();
		$this->assertPattern(
			"~.*?Content-Type: multipart/mixed;\\s* boundary=(\"?)(.*?)\\1\r\n" . //First headers
			".*?\r\n\r\n" . //End of first headers
			".*?\r\n--\\2\r\n" . //First mixed boundary
			"Content-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\3\r\n" . //Second headers
			".*?\r\n\r\n" . //End of second headers
			".*?\r\n--\\4\r\n" . //First alternative boundary
			".*?\r\n--\\4\r\n" .
			".*?\r\n\r\n" .
			".*?\r\n--\\4--" . //End of alternative parts
			"\\s*\r\n--\\2\r\n" . //Second mixed part
			".*?\r\n\r\n" .
			".*?\r\n--\\2--~s", //End of mixed parts
			$structure);
	}
	
	public function testPartsCanBeDetached()
	{
		$msg = new Swift_Message();
		$id1 = $msg->attach(new Swift_Message_Part("foo"));
		$id2 = $msg->attach(new Swift_Message_Part("bar", "text/html"));
		$structure = $msg->build()->readFull();
		
		$this->assertPattern("~^(?:.(?!\r\n\r\n))*?\r\nContent-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\1\r\n" .
			".*?\r\n\r\n.*?\r\n--\\2\r\n" .
			".*?\r\n\r\nfoo\r\n--\\2\r\n" .
			".*?\r\n\r\nbar\r\n" .
			"--\\2--~s", $structure);
		
		$msg->detach($id2);
		$structure = $msg->build()->readFull();
		$this->assertPattern("~^(?:.(?!\r\n\r\n))*?\r\nContent-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\1\r\n" .
			".*?\r\n\r\n.*?\r\n--\\2\r\n" .
			".*?\r\n\r\nfoo\r\n" .
			"--\\2--~s", $structure);
	}
	
	public function testDetachingPartsWithMixedTypesWorks()
	{
		$msg = new Swift_Message();
		$id1 = $msg->attach(new Swift_Message_Part("some part"));
		$id2 = $msg->attach(new Swift_Message_Part("another part"));
		$id3 = $msg->attach(new Swift_Message_Attachment("foobar"));
		$id4 = $msg->attach(new Swift_Message_Attachment("zipbutton"));
		$structure = $msg->build()->readFull();
		$this->assertPattern(
			"~.*?Content-Type: multipart/mixed;\\s* boundary=(\"?)(.*?)\\1\r\n" . //First headers
			".*?\r\n\r\n" . //End of first headers
			".*?\r\n--\\2\r\n" . //First mixed boundary
			"Content-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\3\r\n" . //Second headers
			".*?\r\n\r\n" . //End of second headers
			".*?\r\n--\\4\r\n" . //First alternative boundary
			".*?\r\n--\\4\r\n" . //Next alternative boundary
			".*?\r\n\r\n" .
			".*?\r\n--\\4--" . //End of alternative parts
			"\\s*\r\n--\\2\r\n" . //Second mixed part
			".*?\r\n\r\n" .
			".*?\r\n--\\2\r\n" . //Third mixed part
			".*?\r\n\r\n" .
			".*?\r\n--\\2--~s", //End of mixed parts
			$structure);
		
		$msg->detach($id2);
		$structure = $msg->build()->readFull();
		$this->assertPattern(
			"~.*?Content-Type: multipart/mixed;\\s* boundary=(\"?)(.*?)\\1\r\n" . //First headers
			".*?\r\n\r\n" . //End of first headers
			".*?\r\n--\\2\r\n" . //First mixed boundary
			"Content-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\3\r\n" . //Second headers
			".*?\r\n\r\n" . //End of second headers
			".*?\r\n--\\4\r\n" . //First alternative boundary
			".*?\r\n\r\n" .
			".*?\r\n--\\4--" . //End of alternative parts
			"\\s*\r\n--\\2\r\n" . //Second mixed part
			".*?\r\n\r\n" .
			".*?\r\n--\\2\r\n" . //Third mixed part
			".*?\r\n\r\n" .
			".*?\r\n--\\2--~s", //End of mixed parts
			$structure);
	}
	
	public function testDetachingALLPartsFromAnEmailWithAnAttachmentResultsInCorrectStructure()
	{
		$msg = new Swift_Message();
		$id1 = $msg->attach(new Swift_Message_Part("some part"));
		$id2 = $msg->attach(new Swift_Message_Part("another part"));
		$id3 = $msg->attach(new Swift_Message_Attachment("foobar"));
		$id4 = $msg->attach(new Swift_Message_Attachment("zipbutton"));
		
		$msg->detach($id2);
		$structure = $msg->build()->readFull();
		//$this->dump($structure);
		$this->assertPattern(
			"~.*?Content-Type: multipart/mixed;\\s* boundary=(\"?)(.*?)\\1\r\n" . //First headers
			".*?\r\n\r\n" . //End of first headers
			".*?\r\n--\\2\r\n" . //First mixed boundary
			"Content-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\3\r\n" . //Second headers
			".*?\r\n\r\n" . //End of second headers
			".*?\r\n--\\4\r\n" . //First alternative boundary
			".*?\r\n\r\n" .
			".*?\r\n--\\4--" . //End of alternative parts
			"\\s*\r\n--\\2\r\n" . //Second mixed part
			".*?\r\n\r\n" .
			".*?\r\n--\\2\r\n" . //Third mixed part
			".*?\r\n\r\n" .
			".*?\r\n--\\2--~s", //End of mixed parts
			$structure);
		
		$msg->detach($id1);
		$structure = $msg->build()->readFull();
		//$this->dump($structure);
		$this->assertNoPattern("~Content-Type: multipart/alternative~", $structure);
		$this->assertNoPattern(
			"~.*?Content-Type: multipart/mixed;\\s* boundary=(\"?)(.*?)\\1\r\n" . //First headers
			".*?\r\n\r\n" . //End of first headers
			".*?\r\n--\\2\r\n" . //First mixed boundary
			"Content-Type: multipart/alternative;.*?\r\n" . //This is the kye part we don't want
			".*?\r\n\r\n" .
			".*?\r\n--\\2\r\n" . //Second mixed part
			".*?\r\n\r\n" .
			".*?\r\n--\\2--~s", //End of mixed parts
			$structure);
	}
	
	public function testRemovingALLPartsFromAMessageWithNoAttachmentsIsOk()
	{
		$msg = new Swift_Message();
		$id1 = $msg->attach(new Swift_Message_Part("foo"));
		$id2 = $msg->attach(new Swift_Message_Part("bar", "text/html"));
		$structure = $msg->build()->readFull();
		
		$this->assertPattern("~^(?:.(?!\r\n\r\n))*?\r\nContent-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\1\r\n" .
			".*?\r\n\r\n.*?\r\n--\\2\r\n" .
			".*?\r\n\r\nfoo\r\n--\\2\r\n" .
			".*?\r\n\r\nbar\r\n" .
			"--\\2--~s", $structure);
		
		$msg->detach($id2);
		$structure = $msg->build()->readFull();
		
		$this->assertPattern("~^(?:.(?!\r\n\r\n))*?\r\nContent-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\1\r\n" .
			".*?\r\n\r\n.*?\r\n--\\2\r\n" .
			".*?\r\n\r\nfoo\r\n" .
			"--\\2--~s", $structure);
			
		$msg->detach($id1);
		$structure = $msg->build()->readFull();
		$this->assertNoPattern("~^(?:.(?!\r\n\r\n))*?\r\nContent-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\1\r\n" .
			".*?\r\n\r\n.*?\r\n--\\2~s", $structure);
	}
	
	public function testRemovingALLAttachmentsFromAMixedTypeMessageGivesAMultipartAlternativeType()
	{
		$msg = new Swift_Message();
		$id1 = $msg->attach(new Swift_Message_Part("some part"));
		$id2 = $msg->attach(new Swift_Message_Part("another part"));
		$id3 = $msg->attach(new Swift_Message_Attachment("foobar"));
		$id4 = $msg->attach(new Swift_Message_Attachment("zipbutton"));
		
		$msg->detach($id4);
		$structure = $msg->build()->readFull();
		$this->assertPattern(
			"~.*?Content-Type: multipart/mixed;\\s* boundary=(\"?)(.*?)\\1\r\n" . //First headers
			".*?\r\n\r\n" . //End of first headers
			".*?\r\n--\\2\r\n" . //First mixed boundary
			"Content-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\3\r\n" . //Second headers
			".*?\r\n\r\n" . //End of second headers
			".*?\r\n--\\4\r\n" . //First alternative boundary
			".*?\r\n--\\4\r\n" . //Next alternative boundary
			".*?\r\n\r\n" .
			".*?\r\n--\\4--" . //End of alternative parts
			"\\s*\r\n--\\2\r\n" . //Second mixed part
			".*?\r\n\r\n" .
			".*?\r\n--\\2--~s", //End of mixed parts
			$structure);
		
		$msg->detach($id3);
		$structure = $msg->build()->readFull();
		$this->assertNoPattern("~.*?Content-Type: multipart/mixed~", $structure);
		$this->assertPattern(
			"~.*?Content-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\1\r\n" . //First headers
			".*?\r\n\r\n" . //End of first headers
			".*?\r\n--\\2\r\n" . //First alternative boundary
			".*?\r\n\r\n" . //End of second headers
			".*?\r\n--\\2\r\n" . //Next alternative boundary
			".*?\r\n\r\n" .
			".*?\r\n--\\2--~s", //End of mixed parts
			$structure);
		//$this->dump($structure);
	}
	
	public function testRemovingAllAttachmentsFromAttachmentOnlyMessageLeavesCorrectStructure()
	{
		$msg = new Swift_Message();
		$id1 = $msg->attach(new Swift_Message_Attachment("foo"));
		$id2 = $msg->attach(new Swift_Message_Attachment("bar"));
		$structure = $msg->build()->readFull();
		$this->assertPattern("~^(?:.(?!\r\n\r\n))*?\r\nContent-Type: multipart/mixed;\\s* boundary=(\"?)(.*?)\\1\r\n" .
			".*?\r\n\r\n.*?\r\n--\\2\r\n" .
			".*?\r\n\r\n.*?\r\n--\\2\r\n" .
			".*?\r\n\r\n.*?\r\n" .
			"--\\2--~s", $structure);
		
		$msg->detach($id2);
		$structure = $msg->build()->readFull();
		$this->assertPattern("~^(?:.(?!\r\n\r\n))*?\r\nContent-Type: multipart/mixed;\\s* boundary=(\"?)(.*?)\\1\r\n" .
			".*?\r\n\r\n.*?\r\n--\\2\r\n" .
			".*?\r\n\r\n.*?\r\n" .
			"--\\2--~s", $structure);
			
		$msg->detach($id1);
		$structure = $msg->build()->readFull();
		//$this->dump($structure);
		$this->assertNoPattern("~^(?:.(?!\r\n\r\n))*?\r\nContent-Type: multipart/mixed;\\s* boundary=(\"?)(.*?)\\1\r\n" .
			".*?\r\n\r\n.*?\r\n--\\2~s", $structure);
	}
	
	public function testEmbeddedFilesCreateRelatedMimeType()
	{
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_EmbeddedFile("foo"));
		$this->assertEqual("multipart/related", $msg->getContentType());
		$structure = $msg->build()->readFull();
		$this->assertPattern("~(.(?!\r\n\r\n))*?\r\nContent-Type: multipart/related;\\s* boundary=.*~", $structure);
		
		$msg->attach(new Swift_Message_EmbeddedFile(new Swift_File("../files/manchester.jpeg")));
		$this->assertEqual("multipart/related", $msg->getContentType());
		$structure = $msg->build()->readFull();
		$this->assertPattern("~(.(?!\r\n\r\n))*?\r\nContent-Type: multipart/related;\\s* boundary=.*~", $structure);
	}
	
	public function testEmbeddedFilesWithPartsStillUseRelatedType()
	{
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_Part("foo"));
		$this->assertEqual("multipart/alternative", $msg->getContentType());
		$msg->attach(new Swift_Message_EmbeddedFile("bar"));
		$this->assertEqual("multipart/related", $msg->getContentType());
	}
	
	public function testEmbeddedFilesWithAttachmentsMakesMixedContentType()
	{
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_Part("foo"));
		$this->assertEqual("multipart/alternative", $msg->getContentType());
		$msg->attach(new Swift_Message_EmbeddedFile("bar"));
		$this->assertEqual("multipart/related", $msg->getContentType());
		$msg->attach(new Swift_Message_Attachment("zip"));
		$this->assertEqual("multipart/mixed", $msg->getContentType());
		$msg->attach(new Swift_Message_EmbeddedFile("button"));
		$this->assertEqual("multipart/mixed", $msg->getContentType());
	}
	
	public function testEmbeddedFileStructureIsCorrect()
	{
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_EmbeddedFile("foo"));
		$msg->attach(new Swift_Message_EmbeddedFile("bar"));
		$structure = $msg->build()->readFull();
		//$this->dump($structure);
		$this->assertPattern(
			"~.*?Content-Type: multipart/related;\\s* boundary=(\"?)(.*?)\\1\r\n" . //First headers
			".*?\r\n\r\n" . //End of first headers
			".*?\r\n--\\2\r\n" . //First related boundary
			".*?\r\n\r\n" . //End of second headers
			".*?\r\n--\\2\r\n" . //Next related boundary
			".*?\r\n\r\n" .
			".*?\r\n--\\2--~s", //End of related parts
			$structure);
	}
	
	public function testEmbeddedFileStructureWithMimePartsGivesCorrectStructure()
	{
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_EmbeddedFile("foo"));
		$msg->attach(new Swift_Message_EmbeddedFile("bar"));
		$msg->attach(new Swift_Message_Part("test"));
		$structure = $msg->build()->readFull();
		//$this->dump($structure);
		$this->assertPattern(
			"~.*?Content-Type: multipart/related;\\s* boundary=(\"?)(.*?)\\1\r\n" . //First headers
			".*?\r\n\r\n" . //End of first headers
			".*?\r\n--\\2\r\n" . //First related boundary
			"Content-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\3\r\n" .
			".*?\r\n\r\n" .
			".*?\r\n--\\4\r\n" .
			".*?\r\n\r\n" .
			".*?\r\n--\\4--\r\n\\s*" .
			"--\\2\r\n" .
			".*?\r\n\r\n" . //End of second headers
			".*?\r\n--\\2\r\n" . //Next related boundary
			".*?\r\n\r\n" .
			".*?\r\n--\\2--~s", //End of related parts
			$structure);
	}
	
	public function testEmbeddedFilesWithAttachmentsANDMimePartsGivesCorrectStructure()
	{
		$re = "~.*?Content-Type: multipart/mixed;\\s* boundary=(\"?)(.*?)\\1\r\n" . //First headers
			".*?\r\n\r\n" .
			".*?\r\n--\\2\r\n" . //First mixed boundary
			"Content-Type: multipart/related;\\s* boundary=(\"?)(.*?)\\3\r\n" .
			".*?\r\n\r\n" .
			".*?\r\n--\\4\r\n" . //First related boundary
			"Content-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\5" .
			".*?\r\n\r\n" .
			".*?\r\n--\\6\r\n" . //First alternative boundary
			".*?\r\n--\\6--\r\n\\s*" . //End of alternative parts
			"--\\4\r\n" . //Next related boundary
			".*?\r\n--\\4--\r\n\\s*" . //End of related parts
			"--\\2\r\n" . //next mixed boundary
			".*?\r\n--\\2--~s";
		
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_Part("test"));
		$msg->attach(new Swift_Message_EmbeddedFile("foo"));
		$msg->attach(new Swift_Message_Attachment("bar"));
		
		$structure = $msg->build()->readFull();
		//$this->dump($structure);
		$this->assertPattern($re, $structure);
		
		//Just double-checking by doing it in different orders
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_EmbeddedFile("foo"));
		$msg->attach(new Swift_Message_Part("test"));
		$msg->attach(new Swift_Message_Attachment("bar"));
		
		$structure = $msg->build()->readFull();
		//$this->dump($structure);
		$this->assertPattern($re, $structure);
			
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_Attachment("bar"));
		$msg->attach(new Swift_Message_EmbeddedFile("foo"));
		$msg->attach(new Swift_Message_Part("test"));
		
		$structure = $msg->build()->readFull();
		//$this->dump($structure);
		$this->assertPattern($re, $structure);
			
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_Attachment("bar"));
		$msg->attach(new Swift_Message_Part("test"));
		$msg->attach(new Swift_Message_EmbeddedFile("foo"));
		
		$structure = $msg->build()->readFull();
		//$this->dump($structure);
		$this->assertPattern($re, $structure);
		
		$msg = new Swift_Message();
		$msg->attach(new Swift_Message_Part("test"));
		$msg->attach(new Swift_Message_Attachment("bar"));
		$msg->attach(new Swift_Message_EmbeddedFile("foo"));
		
		$structure = $msg->build()->readFull();
		//$this->dump($structure);
		$this->assertPattern($re, $structure);
			
		$msg = new Swift_Message();
		
		$msg->attach(new Swift_Message_EmbeddedFile("foo"));
		$msg->attach(new Swift_Message_Attachment("bar"));
		$msg->attach(new Swift_Message_Part("test"));
		
		$structure = $msg->build()->readFull();
		//$this->dump($structure);
		$this->assertPattern($re, $structure);
	}
	
	public function testMessageIsCorrectlyStructuredWhenRemovingAllRelatedPartsWhenAllThreeTypesWereMixed()
	{
		$msg = new Swift_Message();
		$id1 = $msg->attach(new Swift_Message_Attachment("bar"));
		$id2 = $msg->attach(new Swift_Message_EmbeddedFile("foo"));
		$id3 = $msg->attach(new Swift_Message_Part("test"));
		
		$msg->detach($id2);
		
		$structure = $msg->build()->readFull();
		//$this->dump($structure);
		$this->assertNoPattern("~Content-Type: multipart/related~", $structure);
		$this->assertPattern(
			"~.*?Content-Type: multipart/mixed;\\s* boundary=(\"?)(.*?)\\1\r\n" . //First headers
			".*?\r\n\r\n" . //End of first headers
			".*?\r\n--\\2\r\n" . //First mixed boundary
			"Content-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\3\r\n" . //Second headers
			".*?\r\n\r\n" . //End of second headers
			".*?\r\n--\\4\r\n" . //First alternative boundary
			".*?\r\n--\\4--" . //End of alternative parts
			"\\s*\r\n--\\2\r\n" . //Second mixed part
			".*?\r\n\r\n" .
			".*?\r\n--\\2--~s", //End of mixed parts
			$structure);
	}
	
	public function testMessageIsCorrectlyStructuredWhenRemovingAllMixedPartsWhenAllThreeTypesWereAdded()
	{
		$msg = new Swift_Message();
		$id1 = $msg->attach(new Swift_Message_Attachment("bar"));
		$id2 = $msg->attach(new Swift_Message_EmbeddedFile("foo"));
		$id3 = $msg->attach(new Swift_Message_Part("test"));
		
		$msg->detach($id1);
		
		$structure = $msg->build()->readFull();
		//$this->dump($structure);
		$this->assertNoPattern("~Content-Type: multipart/mixed~", $structure);
		$this->assertPattern(
			"~.*?Content-Type: multipart/related;\\s* boundary=(\"?)(.*?)\\1\r\n" . //First headers
			".*?\r\n\r\n" . //End of first headers
			".*?\r\n--\\2\r\n" . //First mixed boundary
			"Content-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\3\r\n" . //Second headers
			".*?\r\n\r\n" . //End of second headers
			".*?\r\n--\\4\r\n" . //First alternative boundary
			".*?\r\n--\\4--" . //End of alternative parts
			"\\s*\r\n--\\2\r\n" . //Second mixed part
			".*?\r\n\r\n" .
			".*?\r\n--\\2--~s", //End of mixed parts
			$structure);
	}
	
	public function testMessageIsCorrectlyStructuredWhenRemovingAllAlternativePartsWhenAllThreePartsWereAdded()
	{
		$msg = new Swift_Message();
		$id1 = $msg->attach(new Swift_Message_Attachment("bar"));
		$id2 = $msg->attach(new Swift_Message_EmbeddedFile("foo"));
		$id3 = $msg->attach(new Swift_Message_Part("test"));
		
		$msg->detach($id3);
		
		$structure = $msg->build()->readFull();
		//$this->dump($structure);
		$this->assertNoPattern("~Content-Type: multipart/alternative~", $structure);
		$this->assertPattern(
			"~.*?Content-Type: multipart/mixed;\\s* boundary=(\"?)(.*?)\\1\r\n" . //First headers
			".*?\r\n\r\n" . //End of first headers
			".*?\r\n--\\2\r\n" . //First mixed boundary
			"Content-Type: multipart/related;\\s* boundary=(\"?)(.*?)\\3\r\n" . //Second headers
			".*?\r\n\r\n" . //End of second headers
			".*?\r\n--\\4\r\n" . //First alternative boundary
			".*?\r\n--\\4--" . //End of alternative parts
			"\\s*\r\n--\\2\r\n" . //Second mixed part
			".*?\r\n\r\n" .
			".*?\r\n--\\2--~s", //End of mixed parts
			$structure);
	}
	
	public function testMessageIsCorrectlyStructuredWhenRemovingAllPartsFromARelatedPartMessage()
	{
		$msg = new Swift_Message();
		$id1 = $msg->attach(new Swift_Message_EmbeddedFile("foo"));
		$id2 = $msg->attach(new Swift_Message_Part("test"));
		
		$msg->detach($id2);
		
		$structure = $msg->build()->readFull();
		//$this->dump($structure);
		$this->assertNoPattern("~.*?Content-Type: multipart/alternative~", $structure);
		$this->assertPattern(
			"~.*?Content-Type: multipart/related;\\s* boundary=(\"?)(.*?)\\1\r\n" . //First headers
			".*?\r\n\r\n" . //End of first headers
			".*?\r\n--\\2\r\n" . //First mixed boundary
			".*?\r\n\r\n" .
			".*?\r\n--\\2--~s", //End of mixed parts
			$structure);
	}
	
	public function testMessageIsCorrectlyStructuredWhenRemovingAllRelatedPartsFromARelatedPartMessage()
	{
		$msg = new Swift_Message();
		$id1 = $msg->attach(new Swift_Message_EmbeddedFile("foo"));
		$id2 = $msg->attach(new Swift_Message_Part("test"));
		
		$msg->detach($id1);
		
		$structure = $msg->build()->readFull();
		//$this->dump($structure);
		$this->assertNoPattern("~.*?Content-Type: multipart/related~", $structure);
		$this->assertPattern(
			"~.*?Content-Type: multipart/alternative;\\s* boundary=(\"?)(.*?)\\1\r\n" . //First headers
			".*?\r\n\r\n" . //End of first headers
			".*?\r\n--\\2\r\n" . //First mixed boundary
			".*?\r\n\r\n" .
			".*?\r\n--\\2--~s", //End of mixed parts
			$structure);
	}
	
	public function testMessageIsCorrectlyStructuredWhenRemovingAllRelatedPartsFromAMixedRelatedMessage()
	{
		$msg = new Swift_Message();
		$id1 = $msg->attach(new Swift_Message_EmbeddedFile("foo"));
		$id2 = $msg->attach(new Swift_Message_Attachment("test"));
		
		$msg->detach($id1);
		
		$structure = $msg->build()->readFull();
		//$this->dump($structure);
		$this->assertNoPattern("~.*?Content-Type: multipart/related~", $structure);
		$this->assertPattern(
			"~.*?Content-Type: multipart/mixed;\\s* boundary=(\"?)(.*?)\\1\r\n" . //First headers
			".*?\r\n\r\n" . //End of first headers
			".*?\r\n--\\2\r\n" . //First mixed boundary
			".*?\r\n\r\n" .
			".*?\r\n--\\2--~s", //End of mixed parts
			$structure);
	}
	
	public function testMessageIsCorrectlyStructuredWhenRemovingAllMixedPartsFromAMixedRelatedMessage()
	{
		$msg = new Swift_Message();
		$id1 = $msg->attach(new Swift_Message_EmbeddedFile("foo"));
		$id2 = $msg->attach(new Swift_Message_Attachment("test"));
		
		$msg->detach($id2);
		
		$structure = $msg->build()->readFull();
		//$this->dump($structure);
		$this->assertNoPattern("~.*?Content-Type: multipart/mixed~", $structure);
		$this->assertPattern(
			"~.*?Content-Type: multipart/related;\\s* boundary=(\"?)(.*?)\\1\r\n" . //First headers
			".*?\r\n\r\n" . //End of first headers
			".*?\r\n--\\2\r\n" . //First mixed boundary
			".*?\r\n\r\n" .
			".*?\r\n--\\2--~s", //End of mixed parts
			$structure);
	}
	
	public function testCIDIsReturnedWhenEmbeddingFiles()
	{
		$msg = new Swift_Message();
		$this->assertPattern("/^cid:.+/i", $msg->attach(new Swift_Message_EmbeddedFile("foo")));
		
		$this->assertPattern("/^cid:.+/i", $msg->attach(new Swift_Message_EmbeddedFile("bar")));
	}
}
