<?php

if (!class_exists("FullMockConnection")) Mock::Generate("DummyConnection", "FullMockConnection");
Mock::Generate("Swift_Message", "MockMessage");
Mock::Generate("Swift_Events_SendListener", "MockSendListener");
Mock::Generate("Swift_Connection_SMTP", "MockSMTPConnection");
Mock::GeneratePartial("Swift_Connection_SMTP", "MockSMTPConnectionAuth", array("read", "write", "isAlive", "start", "stop"));
Mock::Generate("Swift_Message_Headers", "MockHeaders");

class TestOfEasySwift extends UnitTestCase
{
	/** Get a mock connection for testing
	 * @param int The number emails you expect to send
	 * @param boolean If the email will be duplicated to all recipients
	 * @return FullMockConnection
	 */
	protected function getWorkingMockConnection($send=1, $duplicate=false, $conn=null)
	{
		$count = 0;
		if ($conn === null) $conn = new FullMockConnection();
		$conn->setReturnValue("isAlive", true);
		$conn->setReturnValueAt($count++, "read", "220 xxx ESMTP");
		$conn->setReturnValueAt($count++, "read", "250-Hello xxx\r\n250-AUTH PLAIN\r\n250 HELP");
		for ($i = 0; $i < $send; $i++)
		{
			$conn->setReturnValueAt($count++, "read", "250 Ok");
			if (!$duplicate || ($i == $send-1))
			{
				$conn->setReturnValueAt($count++, "read", "250 Ok");
				$conn->setReturnValueAt($count++, "read", "354 Go ahead");
				$conn->setReturnValueAt($count++, "read", "250 Ok");
			}
		}
		$conn->setReturnValueAt($count++, "read", "250 Bye");
		return $conn;
	}
	
	public function testNoExceptionsAreThrownBut_HasFailed_IsTrueOnBadConnection()
	{
		$mockConnection = new FullMockConnection();
		$mockConnection->setReturnValue("read", "500 Bad");
		$swift = new EasySwift($mockConnection);
		$this->assertTrue($swift->hasFailed());
		$this->assertFalse($swift->isConnected());
	}
	
	public function testIsConnectedReturnsTrueOnSuccess()
	{
		$mockConnection = $this->getWorkingMockConnection(1);
		
		$swift = new EasySwift($mockConnection);
		$this->assertFalse($swift->hasFailed());
		$this->assertTrue($swift->isConnected());
	}
	
	public function testSendCallsBatchSendWithRecipientListInSwift()
	{
		$mockConnection = $this->getWorkingMockConnection(3);
		$mockConnection->expectMinimumCallCount("write", 13);
		$mockConnection->expectAt(0, "write", array("EHLO xxx", "*"));
		$mockConnection->expectAt(1, "write", array("MAIL FROM: <me@myplace.com>", "*"));
		$mockConnection->expectAt(2, "write", array("RCPT TO: <foo@bar.com>", "*"));
		$mockConnection->expectAt(3, "write", array("DATA", "*"));
		$mockConnection->expectAt(4, "write", array("*", "*"));
		
		$swift = new EasySwift($mockConnection, "xxx");
		$addresses = array("Foo Bar <foo@bar.com>", '"Zip Button" <zip@button.tld>', "mail@cheese.domain");
		$this->assertEqual(3, $swift->send($addresses, "My name <me@myplace.com>", "My subject", "My message"));
	}
	
	public function testSendIsASingleEmailWhenUseExactCopyIsCalled()
	{
		$mockConnection = $this->getWorkingMockConnection(3, true);
		$mockConnection->expectMinimumCallCount("write", 7);
		$mockConnection->expectAt(0, "write", array("EHLO xxx", "*"));
		$mockConnection->expectAt(1, "write", array("MAIL FROM: <me@myplace.com>", "*"));
		$mockConnection->expectAt(2, "write", array("RCPT TO: <foo@bar.com>", "*"));
		$mockConnection->expectAt(3, "write", array("RCPT TO: <zip@button.tld>", "*"));
		$mockConnection->expectAt(4, "write", array("RCPT TO: <mail@cheese.domain>", "*"));
		$mockConnection->expectAt(5, "write", array("DATA", "*"));
		$mockConnection->expectAt(6, "write", array("*", "*"));
		
		$swift = new EasySwift($mockConnection, "xxx");
		$swift->useExactCopy();
		$addresses = array("Foo Bar <foo@bar.com>", '"Zip Button" <zip@button.tld>', "mail@cheese.domain");
		$this->assertEqual(3, $swift->send($addresses, "My name <me@myplace.com>", "My subject", "My message"));
	}
	
	public function testSendIsASingleEmailWhenCcAddressesAreUsed()
	{
		$mockConnection = $this->getWorkingMockConnection(5, true);
		$mockConnection->expectMinimumCallCount("write", 7);
		$mockConnection->expectAt(0, "write", array("EHLO xxx", "*"));
		$mockConnection->expectAt(1, "write", array("MAIL FROM: <me@myplace.com>", "*"));
		$mockConnection->expectAt(2, "write", array("RCPT TO: <foo@bar.com>", "*"));
		$mockConnection->expectAt(3, "write", array("RCPT TO: <zip@button.tld>", "*"));
		$mockConnection->expectAt(4, "write", array("RCPT TO: <mail@cheese.domain>", "*"));
		$mockConnection->expectAt(5, "write", array("RCPT TO: <cc1@address.co.uk>", "*"));
		$mockConnection->expectAt(6, "write", array("RCPT TO: <cc2@address.xxx>", "*"));
		$mockConnection->expectAt(7, "write", array("DATA", "*"));
		$mockConnection->expectAt(8, "write", array("*", "*"));
		
		$swift = new EasySwift($mockConnection, "xxx");
		$swift->addCc("Carbon Copy Recipient One <cc1@address.co.uk>");
		$swift->addCc("cc2@address.xxx");
		$addresses = array("Foo Bar <foo@bar.com>", '"Zip Button" <zip@button.tld>', "mail@cheese.domain");
		$this->assertEqual(5, $swift->send($addresses, "My name <me@myplace.com>", "My subject", "My message"));
	}
	
	public function testAddingPartsIsHandledByMessageObject()
	{
		$mockConnection = $this->getWorkingMockConnection();
		
		$swift = new EasySwift($mockConnection, "xxx");
		$mockMessage = new MockMessage();
		$mockMessage->expectOnce("attach");
		$swift->newMessage($mockMessage);
		$swift->addPart("my part");
	}
	
	public function testAddingAttachmentsIsHandledByMessageObject()
	{
		$mockConnection = $this->getWorkingMockConnection();
		
		$swift = new EasySwift($mockConnection, "xxx");
		$mockMessage = new MockMessage();
		$mockMessage->expectCallCount("attach", 2);
		$swift->newMessage($mockMessage);
		$swift->addAttachment("my attachment", "my name.txt");
		$swift->addAttachment(new Swift_File("../files/gecko.png"));
	}
	
	public function testAddingImagesIsHandledByMessageObject()
	{
		$mockConnection = $this->getWorkingMockConnection();
		
		$swift = new EasySwift($mockConnection, "xxx");
		$mockMessage = new MockMessage();
		$mockMessage->expectCallCount("attach", 3);
		$swift->newMessage($mockMessage);
		$swift->addImage("../files/manchester.jpeg");
		$swift->addImage("../files/durham.gif");
		$swift->addImage("../files/gecko.png");
	}
	
	public function testCIDSrcValueIsReturnedWhenAddingImage()
	{
		$mockConnection = $this->getWorkingMockConnection();
		
		$swift = new EasySwift($mockConnection, "xxx");
		
		$this->assertPattern("/^cid:.+\$/i", $swift->addImage("../files/manchester.jpeg"));
		$this->assertPattern("/^cid:.+\$/i", $swift->addImage("../files/durham.gif"));
		$this->assertPattern("/^cid:.+\$/i", $swift->addImage(new Swift_File("../files/gecko.png")));
	}
	
	public function testCIDSrcValueIsReturnedWhenAddingEmbeddedFile()
	{
		$mockConnection = $this->getWorkingMockConnection();
		
		$swift = new EasySwift($mockConnection, "xxx");
		
		$this->assertPattern("/^cid:.+\$/i", $swift->embedFile(file_get_contents("../files/manchester.jpeg"), "image/jpeg"));
		$this->assertPattern("/^cid:.+\$/i", $swift->embedFile(new Swift_File("../files/durham.gif"), "image/gif", "myimage.gif"));
		$this->assertPattern("/^cid:my_cid123\$/i", $swift->embedFile(file_get_contents("../files/gecko.png"), "image/png", "myimage.png", "my_cid123"));
	}
	
	public function testPluginsAreLoadedWithEasySwiftButHandledBySwiftAsNormal()
	{
		$conn = $this->getWorkingMockConnection();
		$swift = new EasySwift($conn);
		$plugin = new MockSendListener();
		$plugin->expectOnce("sendPerformed");
		$swift->loadPlugin($plugin, "myplugin");
		$swift->send("foo@bar.com", "me@mydomain.com", "Subject", "body");
	}
	
	public function testSmtpAuthenticatorsAreAddedIfSmtpConnectionIsUsed()
	{
		$conn = $this->getWorkingMockConnection(1, false, new MockSMTPConnection());
		$auth = new Swift_Authenticator_PLAIN();
		$conn->expectOnce("attachAuthenticator", array($auth));
		
		$swift = new EasySwift($conn);
		$swift->loadAuthenticator($auth);
	}
	
	public function testSMTPAuthenticationReturnsTrueOnSuccess()
	{
		$conn = new MockSMTPConnectionAuth();
		$conn->setReturnValue("isAlive", true);
		$conn->setReturnValueAt(0, "read", "220 xxx ESMTP");
		$conn->setReturnValueAt(1, "read", "250-Hello xxx\r\n250-AUTH PLAIN\r\n250 HELP");
		$conn->setReturnValueAt(2, "read", "235 Authenticated");
		$conn->expectAt(1, "write", array("AUTH PLAIN foo\0foo\0bar", "*"));
		
		$auth = new Swift_Authenticator_PLAIN();
		
		$swift = new EasySwift($conn);
		$swift->loadAuthenticator($auth);
		$this->assertTrue($swift->authenticate("foo", "bar"));
	}
	
	public function testSMTPAuthenticationReturnsFalseOnFailure()
	{
		$conn = new MockSMTPConnectionAuth();
		$conn->setReturnValue("isAlive", true);
		$conn->setReturnValueAt(0, "read", "220 xxx ESMTP");
		$conn->setReturnValueAt(1, "read", "250-Hello xxx\r\n250-AUTH PLAIN\r\n250 HELP");
		$conn->setReturnValueAt(2, "read", "500 No good");
		$conn->expectAt(1, "write", array("AUTH PLAIN foo\0foo\0bar", "*"));
		
		$auth = new Swift_Authenticator_PLAIN();
		
		$swift = new EasySwift($conn);
		$swift->loadAuthenticator($auth);
		$this->assertFalse($swift->authenticate("foo", "bar"));
	}
	
	public function testMessageHeadersAreInvokedWhenAddHeadersIsCalled()
	{
		$conn = $this->getWorkingMockConnection();
		$swift = new EasySwift($conn);
		$headers = new MockHeaders();
		$headers->expectCallCount("set", 2);
		$headers->expectAt(0, "set", array("Foo", "test"));
		$headers->expectAt(1, "set", array("Bar", "test2"));
		$swift->message->setHeaders($headers);
		$swift->addHeaders("Foo: test\r\nBar: test2");
	}
	
	public function testAttributesCanBeSetInHeaders()
	{
		$conn = $this->getWorkingMockConnection();
		$swift = new EasySwift($conn);
		$headers = new MockHeaders();
		$headers->expectCallCount("set", 2);
		$headers->expectAt(0, "set", array("Foo", "test"));
		$headers->expectAt(1, "set", array("Bar", "test2"));
		$headers->expectCallCount("setAttribute", 4);
		$headers->expectAt(0, "setAttribute", array("Foo", "xxx", "yyy"));
		$headers->expectAt(1, "setAttribute", array("Bar", "abc", "def"));
		$headers->expectAt(2, "setAttribute", array("Bar", "example", "something"));
		$headers->expectAt(3, "setAttribute", array("Bar", "foo", "bar"));
		$swift->message->setHeaders($headers);
		$swift->addHeaders("Foo: test; xxx=\"yyy\"\r\nBar: test2;\r\n abc=def; example=\"something\"; foo=bar");
	}
}
